******************************************************************************
* FORTRAN subroutine for the calculation of the deflection and magnification 
* by a family of general power-law ellipsoidal mass distributions, based on 
* the Fourier series coefficient functions (the "I-functions") given in Chae, 
* Khersonsky, & Turnshek (1998, ApJ, 506, 80) and Chae (2002). 
* 
* The I-functions for r > r_u [:= r_c/SQRT(1-e)] and r < r_u are different;
* so, the two cases are calculated separately. For r > r_u, this code
* allows one to calculate the deflection angle and magnification matrix 
* components up to `arbitrarily' small errors by controlling the truncation 
* of the series. Also, one can speed up the calculation by increasing the 
* required calculational errors, which should be determined by the user for
* the problem under consideration. Similarly for r < r_u, the code allows
* fast and accurate calculation; however, for very large ellipticities (> 0.8)
* the smallest possible calculational error is about 10^(-6) and the 
* calculation slows down significantly.
*
* The assumed form of the 2-dimensional mass distribution in units of 
* Sigma_crit (where Sigma_crit is the critical surface mass density) is given 
* as follows:
*
*                                 K_0
*    K(r,phi) = -----------------------------------------------------
*               { 1 + (r/r_c)^2 [1 + e cos 2(phi-phi_0)] }^[(nu-1)/2]
*
* Here 
*      K_0 (``kappa zero''): dimensionless surface density at r=0  => prmt(1)
*      nu : radial index such that the de-projected 3-dimensinal density 
*           rho(r) is proportional to r^(-nu) at large r/r_c. 1 < nu < 3  
*           (nu = 2: isothermal profile) => prmt(2)
*      r_c : core radius  => prmt(3)
*      e : (non-negative and < 1) parameter related to the ellipticity via 
*          ellipticity = 1 - SQRT((1 - e)/(1 + e)).
*          The ellipticity (0 =/< ellip < 1) is prmt(4) not e for the code.
*      phi_0 : Standard position angle (north through east) => prmt(5) 
*      x & y coordinates of the image position  => prmt(6) & prmt(7)
*      Finally, prmt(8) is the parameter controlling the truncation of the 
*      series. 
*      Note: The errors for the calculated deflection and magnification      
*      components are smaller than the given value of prmt(8). In particular,
*      for relatively low ellipticities (ellipticity < 0.4), the errors are 
*      smaller than prmt(8) by orders of magnitude.                          
*
* The outputs are as follows:
*
*     dfl(1) & dfl(2): x & y components of the deflection angle
*     mgnf(1) = 1 - psi_(,11)
*     mgnf(2) = -psi_(,12)
*     mgnf(3) = -psi_(,21) = mgnf(2)
*     mgnf(4) = 1 - psi_(,22)
*     mgnf(5) = Magnification factor = [mgnf(1)*mgnf(4)-mgnf(2)^2]^(-1)
*
* Last modified on 11 August 2001, Kyu-Hyun Chae
* Please send your comments to chae@jb.man.ac.uk or chae@phyast.pitt.edu   
******************************************************************************
      SUBROUTINE spleFS(prmt,dfl,mgnf)
* spleFS stands for softened power-law ellipsoid Fourier Series.
* Uses hg2F1, Int2m1, Icos_km.
* "I-functions" are denoted by the notations "Int*" where Int means Integral.
      Implicit none
      Integer m,m1,m2,m3,mmax,k,k0,kmax,kmax2,l,i,j,n
      Parameter(n=200)
      Double Precision prmt(8),dfl(2),mgnf(5)
      Double Precision kappa0,nu,rc,ellip,e,phi0,eps,mu,P,Q,r,ru,rd
      Double Precision phi,phi1,pi,x,y,h,e1,e1sqrt,e2,e2mu,f1,f2
      Double Precision Int0,Int0p,Int1(n),Int2(n),Int3(n),Int2m1
      Double Precision ALr,ALphi,ALx,ALy,ALrr,ALrp,ALpr,ALpp
      Double Precision PSI11,PSI12,PSI21,PSI22,cphi,sphi,zeta 
      Double Precision hg2F1,Fval_muk0(0:1000),Fval_muk0a(0:1000)
      Double Precision Fval_mum(0:1000),hgt1,hgt2,hgt3,a,b,c
      Double Precision hgt1p,hgt2p,hgt3p,ap,bp,cp,t,t1,t2,t3,et
      Double Precision sum0,sum,sum0_2,sum_2,sum0_11,sum_11,sum0_12
      Double Precision sum_12,sum0_int1,sum_int1,sum0_int2,sum_int2
      Double Precision fracmax,gamma1,gamma1p,gamma2,gamma3,gamma4
      Double Precision gamma5,gamma6,gamma7,muinv,dpm,dpk,dpl
      Double Precision Sn(0:n),Cn(0:n),An(0:n),Bn(0:n),S0,Cnsum,Snsum
      Double Precision cphi1(n),sphi1(n),rat,Icos_km
      Parameter(pi=3.14159265359d0)
      External hg2F1,Int2m1,Icos_km
*
      kappa0=prmt(1)
      nu=prmt(2)
      rc=prmt(3)
      ellip=MIN(abs(prmt(4)),0.95d0)
      phi0=prmt(5)
      x=prmt(6)
      y=prmt(7)
      eps=prmt(8)
      mu=(nu-3.d0)/2.d0
      muinv=1.d0/mu
      e=ABS(ellip*(2.d0-ellip)/(1.d0+(1.d0-ellip)**2))
      P=1.d0/rc**2
      Q=P*e
      if (abs(x/rc) .lt. 1.d-7 .and. abs(y/rc) .lt. 1.d-7) then
         r=1.d-7*rc
      else
         r=sqrt(x**2+y**2)
      end if
      phi=datan2(y,x)
*
* Lensing by a circularly symmetrical deflector
*----------------------------------------------
      IF (e .lt. 1.d-6) THEN
         cphi=cos(phi)
         sphi=sin(phi)
         ALr=kappa0*(1.d0-1.d0/(1.d0+P*r**2)**mu)/(mu*P*r)
         ALx=ALr*cphi
         ALy=ALr*sphi
         ALrr=2.d0*kappa0/(1.d0+P*r**2)**(mu+1.d0)-ALr/r
         PSI11=cphi**2*ALrr+sphi**2*ALr/r
         PSI22=sphi**2*ALrr+cphi**2*ALr/r
         PSI12=sphi*cphi*ALrr-sphi*cphi*ALr/r
         PSI21=PSI12          
         go to 100
      END IF
*
* Hereafter:lensing by an elliptical mass distribution
*-----------------------------------------------------    
      f1=1.d0+P*r**2
      f2=sqrt(f1**2-(Q*r**2)**2)      
      h=r/f2
      e1=abs((f1-f2)/(f1+f2))
      e1sqrt=sqrt(e1)
      e2=0.5d0*(f1/f2**2+1.d0/f2)
      e2mu=e2**mu
      ru=rc/SQRT(1.d0-e)
      rd=rc/SQRT(1.d0+e)
*---------------------------------------------------------*
* Evaluation of F(m-mu,-mu;m+1;e1) & F(m-1-mu,-mu;m+1;e1) *
* for m = 1 to mmax.                                      *
*---------------------------------------------------------*
* Determine m3 & mmax:
      gamma1=mu+1.d0
      t1=e1sqrt
      sum=gamma1*e1sqrt
      m=1
 3    sum0=sum
      m=m+1
      dpm=DBLE(m)
      gamma1=gamma1*(1.d0+mu/dpm)
      t1=t1*e1sqrt
      sum=sum+gamma1*t1
      if (ABS((sum-sum0)/sum0) .gt. eps) go to 3
      m3=m     
      mmax=m3+IDINT(m3*0.3d0)+1
*
      dpm=DBLE(mmax)
      hgt1=hg2F1(dpm+1.d0-mu,-mu,dpm+2.d0,e1,eps)
      hgt2=hg2F1(dpm-mu,-mu,dpm+2.d0,e1,eps)
      a=dpm+1.d0-mu
      b=-mu
      c=dpm+2.d0
      do m=mmax,0,-1
      hgt3=hgt1*(a-1.d0)/(c-1.d0)+hgt2*(c-a)/(c-1.d0)
      hgt2=hgt2*e1*(c-b-1.d0)/(c-1.d0)+(1.d0-e1)*hgt3
      hgt1=hgt3
      a=a-1.d0
      c=c-1.d0
      Fval_muk0(m)=hgt3
      Fval_muk0a(m)=hgt2
      enddo
*
*--------------------------*
* Evaluation of Int0-prime *
*--------------------------*
      Int0p=2.d0*h*e2mu*Fval_muk0(0)
*
*--------------------*
* Evaluation of Int0 *
*--------------------*
* r > r_u
      IF (r .gt. 1.3d0*ru) THEN
      hgt1=-Fval_muk0a(0)*(1.d0+mu)*muinv/(1.d0-e1)
     a     +Fval_muk0(0)*(2.d0*mu+1.d0)*muinv/(1.d0-e1)
      hgt2=Fval_muk0(0)
      a=-mu+1.d0
      b=-mu
      c=1.d0
      sum=hgt2
      t=1.d0
      k=0
 10   sum0=sum
      k=k+1
      hgt3=(a-1.d0)/(c-b)*(1.d0-e1)*hgt1
     a    +((b-1.d0)*(c-b)-(a-1.d0)*(c-a))/((b-a)*(c-b))*hgt2
      hgt2=(c+2.d0-2.d0*a-(b-a)*e1)/(c-a+1.d0)*hgt3
     a    -((a-1.d0)*(b-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)**2*hgt1
     a    +((a-1.d0)*(c-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)*hgt2
      hgt1=hgt3
      a=a-1.d0
      b=b-1.d0
      t=t*e2
      sum=sum+t*hgt2
      if (abs((sum-sum0)/sum) .gt. eps) go to 10
      Int0=muinv*(1.d0/(r*sqrt(P**2-Q**2))-h*e2mu*sum)
* r < r_u
      ELSE
      m=0
      dpm=0.d0
      t=(r/rc)**2/(1.d0+(r/rc)**2)
      et=e*t
      t1=1.d0-t
      kmax=MAX(INT(LOG10(eps)/LOG10(et)),2)
      dpk=DBLE(kmax+m)
      hgt1=hg2F1(dpk+mu+1.d0,1.d0,dpk+dpm+2.d0,t,eps)
      hgt2=hg2F1(dpk+mu,1.d0,dpk+dpm+2.d0,t,eps)
      Fval_mum(kmax+m)=hgt1
      a=dpk+mu+1.d0
      b=1.d0
      c=dpk+dpm+2.d0
      do k=kmax+m-1,m,-1
      hgt3=hgt1*(a-1.d0)/(c-1.d0)+hgt2*(c-a)/(c-1.d0)
      hgt2=hgt2*t*(c-b-1.d0)/(c-1.d0)+(1.d0-t)*hgt3
      hgt1=hgt3
      a=a-1.d0
      c=c-1.d0
      Fval_mum(k)=hgt1
      enddo
      gamma2=1.d0
      t2=Icos_km(m,m)
      t3=1.d0
      sum=(gamma2/(2.d0*dpm+1.d0))*2.d0*t2*Fval_mum(m)
      k=0
 11   k=k+1
      sum0=sum
      dpk=DBLE(k)
      t3=t3*et
      gamma2=gamma2*(mu/(dpk+dpm)+1.d0)
      if (1+(-1)**k .eq. 0) then
        go to 11
      else
        t2=gamma2*t3*2.d0*Icos_km(k+m,m)
        sum=sum+(t2/(dpk+2.d0*dpm+1.d0))*Fval_mum(m+k)
        if (ABS((sum-sum0)/sum) .gt. eps) go to 11
      endif
      t2=(r/pi)*t1**(mu+1.d0)
      Int0=t2*sum
      END IF
*
*---------------------------------------------------*
* Evaluation of Int3: keep Int3(1) through Int3(m3) *
*---------------------------------------------------*
      gamma1=1.d0
      do m=1,m3
      dpm=DBLE(m)
      gamma1=gamma1*(1.d0+mu/dpm)
      Int3(m)=(-e1sqrt)**m*h*e2mu*gamma1*Fval_muk0(m)
      enddo
*
*------------------------------------------------------------*
* Evaluation of Int1 and Int2: keep Int1(1) through Int1(m1) *
*                              keep Int2(1) through Int2(m1) *
*------------------------------------------------------------*
* r > r_u
      IF (r .gt. 1.3d0*ru) THEN
      m=0
      gamma1=muinv
      gamma3=1.d0
      gamma4=muinv
      gamma7=1.d0
      sum_int1=0.d0
      sum_int2=0.d0
 20   sum0_int1=sum_int1
      sum0_int2=sum_int2
      m=m+1
      dpm=DBLE(m)
      gamma1=gamma1*e1sqrt*(dpm-1.d0+mu)/dpm
      gamma3=gamma3*(dpm+mu)*(-dpm+mu+1.d0)
      gamma4=gamma4/(dpm*(mu-dpm))
      gamma7=gamma7*(2*m*(2*m-1))
      if (m .le. mmax) then
         hgt1=Fval_muk0(m)
         hgt2=Fval_muk0a(m)
      else
         hgt1=hg2F1(dpm-mu,-mu,dpm+1.d0,e1,eps)
         hgt2=hg2F1(dpm-1.d0-mu,-mu,dpm+1.d0,e1,eps)
         mmax=m
      end if
      hgt1p=1.d0
      hgt2p=1.d0+e1*dpm/(dpm+1.d0)
      gamma2=1.d0
      gamma5=gamma3
      gamma6=gamma7
      a=dpm-mu
      b=-mu
      c=dpm+1.d0
      ap=0.d0
      bp=-dpm
      cp=dpm+1.d0
      sum_2=hgt1
      sum_11=gamma3*hgt1
      sum_12=gamma7*hgt1p
      k=0
 21   sum0_2=sum_2
      sum0_11=sum_11
      k=k+1
      dpk=DBLE(k)
      hgt3=(a-1.d0)/(c-b)*(1.d0-e1)*hgt1
     a    +((b-1.d0)*(c-b)-(a-1.d0)*(c-a))/((b-a)*(c-b))*hgt2
      hgt2=(c+2.d0-2.d0*a-(b-a)*e1)/(c-a+1.d0)*hgt3
     a    -((a-1.d0)*(b-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)**2*hgt1
     a    +((a-1.d0)*(c-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)*hgt2
      hgt1=hgt3
      gamma2=gamma2*e2
      gamma5=gamma5*((dpk+dpm+mu)/(dpk-dpm+mu))*e2
      a=a-1.d0
      b=b-1.d0
      sum_2=sum_2+gamma2*hgt3
      sum_11=sum_11+gamma5*hgt3
      fracmax=MAX(ABS((sum_2-sum0_2)/sum_2)
     a           ,ABS((sum_11-sum0_11)/sum_11))
      if (fracmax .gt. eps) go to 21
      Int2(m)=(-1.d0)**m*h*e2mu*gamma1*sum_2
      if (ABS(e2**(dpm-mu)*gamma7/gamma3) .gt. eps) then
      k=0
 22   sum0_12=sum_12
      k=k+1
      dpk=DBLE(k)
      hgt3p=(ap-1.d0)/(cp-bp)*(1.d0-e1)*hgt1p
     a +((bp-1.d0)*(cp-bp)-(ap-1.d0)*(cp-ap))/((bp-ap)*(cp-bp))*hgt2p
      hgt2p=(cp+2.d0-2.d0*ap-(bp-ap)*e1)/(cp-ap+1.d0)*hgt3p
     a  -((ap-1.d0)*(bp-ap))/((cp-ap+1.d0)*(cp-bp))*(1.d0-e1)**2*hgt1p
     a  +((ap-1.d0)*(cp-ap))/((cp-ap+1.d0)*(cp-bp))*(1.d0-e1)*hgt2p
      hgt1p=hgt3p
      gamma6=gamma6*((dpk+2.d0*dpm)/dpk)*e2
      ap=ap-1.d0
      bp=bp-1.d0
      sum_12=sum_12+gamma6*hgt3p
      if (abs((sum_12-sum0_12)/sum_12) .gt. eps) go to 22
      end if
      Int1(m)=-h*(-e1sqrt)**m*gamma4*(e2mu*sum_11-e2**m*sum_12)
      sum_int2=sum_int2+dpm*Int2(m)
      sum_int1=sum_int1+dpm*Int1(m)
      fracmax=MAX(ABS((sum_int2-sum0_int2)/sum_int2)
     a           ,ABS((sum_int1-sum0_int1)/sum_int1))
      if (fracmax .gt. eps) go to 20
      m1=m
      m2=m
*
* r < r_u
      ELSE IF (r .gt. 0.5d0*rd) THEN
      t=(r/rc)**2/(1.d0+(r/rc)**2)
      et=e*t
      t1=1.d0-t
      m=0
      gamma1=1.d0
      sum_int1=0.d0
      sum_int2=0.d0
 30   m=m+1
      sum0_int1=sum_int1
      sum0_int2=sum_int2
      dpm=DBLE(m)
      k0=INT(mu*e/(1.d0-e))-(m+1)
      if (k0 .lt. 1) then
        kmax=MAX(INT(LOG10(eps)/LOG10(e*t)),2)
        kmax2=MAX(INT(LOG10(eps)/LOG10(e)),2)
      else
        kmax=k0+INT(LOG10(eps)/LOG10((e*t+1.d0)/2.d0))
        kmax2=k0+INT(LOG10(eps)/LOG10((e+1.d0)/2.d0))
      endif
      gamma1=(1.d0+mu/dpm)*gamma1
      dpk=DBLE(kmax+m)
      hgt1=hg2F1(dpk+mu+1.d0,1.d0,dpk+dpm+2.d0,t,eps)
      hgt2=hg2F1(dpk+mu,1.d0,dpk+dpm+2.d0,t,eps)
      Fval_mum(kmax+m)=hgt1
      a=dpk+mu+1.d0
      b=1.d0
      c=dpk+dpm+2.d0
      do k=kmax+m-1,m,-1
      hgt3=hgt1*(a-1.d0)/(c-1.d0)+hgt2*(c-a)/(c-1.d0)
      hgt2=hgt2*t*(c-b-1.d0)/(c-1.d0)+(1.d0-t)*hgt3
      hgt1=hgt3
      a=a-1.d0
      c=c-1.d0
      Fval_mum(k)=hgt1
      enddo
      gamma2=gamma1
      t2=Icos_km(m,m)
      t3=1.d0
      sum=(gamma2/(2.d0*dpm+1.d0))*2.d0*t2*Fval_mum(m)
      sum_2=gamma2*2.d0*t2
      k=0
 31   k=k+1
      sum0=sum
      sum0_2=sum_2
      dpk=DBLE(k)
      t3=t3*et
      gamma2=gamma2*(mu/(dpk+dpm)+1.d0)
      if (k .eq. 1) then
        hgt1p=1.d0
        hgt2p=1.d0-t1/(1.d0+1.d0/(mu+dpm))
        ap=-1.d0
        bp=mu+dpm
        cp=mu+dpm+1.d0
      else
        hgt3p=(ap*(1.d0-t1)/(cp-ap))*hgt1p
     a        -((2.d0*ap-cp-ap*t1+bp*t1)/(cp-ap))*hgt2p
        hgt1p=hgt2p
        hgt2p=hgt3p
        ap=ap-1.d0
      endif
      if (1+(-1)**k .eq. 0) then
        go to 31
      else
        t2=gamma2*t3*2.d0*Icos_km(k+m,m)
        if (k .le. kmax) 
     a    sum=sum+(t2/(dpk+2.d0*dpm+1.d0))*Fval_mum(m+k)
        sum_2=sum_2+(t2/t**k)*hgt2p
        fracmax=MAX(ABS((sum-sum0)/sum),ABS((sum_2-sum0_2)/sum_2))
        if (k .le. kmax2) then
        if (fracmax .gt. eps) go to 31
        endif
      endif
      t2=(r/pi)*(-t*e)**m*t1**(mu+1.d0)
      Int1(m)=t2*sum
      Int2(m)=(t2/(mu+dpm))*sum_2/t
      sum_int1=sum_int1+ABS(dpm*Int1(m))
      sum_int2=sum_int2+ABS(dpm*Int2(m))
      fracmax=MAX(ABS((sum_int1-sum0_int1)/sum_int1)
     a            ,ABS((sum_int2-sum0_int2)/sum_int2))
      if (fracmax .gt. eps) go to 30
      m1=m
      m2=m
*
* r << r_d
      ELSE
* Int1(m):
*--------
      t=(r/rc)**2/(1.d0+(r/rc)**2)
      et=e*t
      t1=1.d0-t
      m=0
      gamma1=1.d0
      sum_int1=0.d0
 40   m=m+1
      sum0_int1=sum_int1
      dpm=DBLE(m)
      k0=INT(mu*e/(1.d0-e))-(m+1)
      if (k0 .lt. 1) then
        kmax=MAX(INT(LOG10(eps)/LOG10(e*t)),2)
      else
        kmax=k0+INT(LOG10(eps)/LOG10((e*t+1.d0)/2.d0))
      endif
      gamma1=(1.d0+mu/dpm)*gamma1
      dpk=DBLE(kmax+m)
      hgt1=hg2F1(dpk+mu+1.d0,1.d0,dpk+dpm+2.d0,t,eps)
      hgt2=hg2F1(dpk+mu,1.d0,dpk+dpm+2.d0,t,eps)
      Fval_mum(kmax+m)=hgt1
      a=dpk+mu+1.d0
      b=1.d0
      c=dpk+dpm+2.d0
      do k=kmax+m-1,m,-1
      hgt3=hgt1*(a-1.d0)/(c-1.d0)+hgt2*(c-a)/(c-1.d0)
      hgt2=hgt2*t*(c-b-1.d0)/(c-1.d0)+(1.d0-t)*hgt3
      hgt1=hgt3
      a=a-1.d0
      c=c-1.d0
      Fval_mum(k)=hgt1
      enddo
      gamma2=gamma1
      t2=Icos_km(m,m)
      t3=1.d0
      sum=(gamma2/(2.d0*dpm+1.d0))*2.d0*t2*Fval_mum(m)
      k=0
 41   k=k+1
      sum0=sum
      dpk=DBLE(k)
      t3=t3*et
      gamma2=gamma2*(mu/(dpk+dpm)+1.d0)
      if (1+(-1)**k .eq. 0) then
        go to 41
      else
        t2=gamma2*t3*2.d0*Icos_km(k+m,m) 
        sum=sum+(t2/(dpk+2.d0*dpm+1.d0))*Fval_mum(m+k)
        if (k .le. kmax) then
        if (ABS((sum-sum0)/sum) .gt. eps) go to 41
        endif
      endif
      t2=(r/pi)*(-t*e)**m*t1**(mu+1.d0)
      Int1(m)=t2*sum
      sum_int1=sum_int1+ABS(dpm*Int1(m))
      fracmax=ABS((sum_int1-sum0_int1)/sum_int1)
      if (fracmax .gt. eps) go to 40
      m1=m
* Int2(m):
*--------
      Int2(1)=Int2m1(mu,P,Q,r,eps)
      m=1
      gamma1p=e1sqrt
      sum_int2=ABS(Int2(1))
 45   sum0_int2=sum_int2
      m=m+1
      dpm=DBLE(m)
      gamma1p=gamma1p*e1sqrt*(dpm-1.d0+mu)/dpm
      If (LOG10(e1) .lt. -20.d0) then
         sum_2=1.d0/(1.d0-e2)
      Else if (LOG10(e1) .lt. -10.d0) then
         sum_2=(1.d0/(1.d0-e2))
     a          *(1.d0+(e1/(dpm+1.d0))*(1.d0-mu*(dpm-mu)
     a                  -((dpm-2.d0*mu)*e2+1.d0)/(1.d0-e2)
     a                             +2.d0*e2/(1.d0-e2)**2))
      Else
      S0=1.d0/(1.d0-e2)
      sum_2=S0
      Sn(0)=S0-1.d0
      a=dpm-mu
      b=-mu
      c=dpm+1.d0
      t=1.d0
      An(0)=1.d0
      Bn(0)=1.d0
      l=0
 46   sum0_2=sum_2
      l=l+1
      dpl=DBLE(l)
      t=t*e1/((c+dpl-1.d0)*dpl)
      if (l .eq. 1) then
         a=-a
         b=-b
         An(0)=An(0)*a
         Bn(0)=Bn(0)*b
         An(1)=1.d0
         Bn(1)=1.d0
      else
         a=a-1.d0
         b=b-1.d0
         do j=l-1,1,-1
         An(j)=a*An(j)+An(j-1)
         Bn(j)=b*Bn(j)+Bn(j-1)
         enddo
         An(0)=An(0)*a
         Bn(0)=Bn(0)*b
         An(l)=1.d0
         Bn(l)=1.d0
      endif
      do j=0,2*l
         Cnsum=0.d0
         do i=0,l
         if (j-i .ge. 0 .and. j-i .le. l) then
            Cnsum=Cnsum+An(i)*Bn(j-i)
         endif
         enddo
         Cn(j)=Cnsum
      enddo
      if (l .eq. 1) then
         Sn(1)=S0*Sn(0)
         Sn(2)=-S0*(Sn(0)-2.d0*Sn(1))
      else
         do j=2*l-1,2*l
           gamma1=DBLE((-1)**j)
           Snsum=gamma1*Sn(0)
           do i=1,j-1
           gamma1=-gamma1*DBLE(j-i+1)/DBLE(i)
           Snsum=Snsum+gamma1*Sn(i)
           enddo
           Sn(j)=-S0*Snsum
         enddo
      endif
      sum_2=sum_2+t*Cn(0)*S0
      do j=1,2*l
      sum_2=sum_2+t*Cn(j)*Sn(j)
      enddo
      if (l .le. 20) then
      if (abs((sum_2-sum0_2)/sum_2) .gt. eps) go to 46
      endif
      End if
      Int2(m)=DBLE((-1)**m)*h*e2mu*gamma1p*sum_2
      sum_int2=sum_int2+ABS(dpm*Int2(m))
      fracmax=ABS((sum_int2-sum0_int2)/sum_int2)
      if (m .le. 2) then
        go to 45
      else
        rat=ABS(Int2(m)/Int2(m-1))
        if (rat .lt. 1.d0) then
          if (fracmax .gt. eps) go to 45
          m2=m
        else
          Int2(m)=0.d0
          m2=m-1
        endif
      endif
      END IF
*------------------------------------------------*
* Calculation of the Deflection Angle Components *
*------------------------------------------------*
      if (m1 .gt. m2) then
        do m=m2+1,m1
        Int2(m)=0.d0
        enddo
      else if (m2 .gt. m1) then
        do m=m1+1,m2
        Int1(m)=0.d0
        enddo
        m1=m2
      endif
      phi1=2.d0*(phi-phi0)
      cphi=cos(phi)
      sphi=sin(phi)
      cphi1(1)=cos(phi1)
      sphi1(1)=sin(phi1)
      if (m1 .gt. 1) then
      do m=2,MAX(m1,m3)
      cphi1(m)=cphi1(1)*cphi1(m-1)-sphi1(1)*sphi1(m-1)
      sphi1(m)=sphi1(1)*cphi1(m-1)+cphi1(1)*sphi1(m-1)
      enddo
      end if
*:Alpha(r) and Alpha(phi)
      ALr=Int0
      ALphi=0.d0
      do m=1,m1
      ALr=ALr+cphi1(m)*(Int1(m)-Int2(m))
      ALphi=Alphi+sphi1(m)*(Int1(m)+Int2(m))
      enddo
      ALr=kappa0*ALr
      ALphi=kappa0*ALphi
*:transformation to Alpha(x) and Alpha(y)
      ALx=ALr*cphi-ALphi*sphi
      ALy=ALr*sphi+ALphi*cphi
*----------------------------------------*
*Calculation of the magnification matrix *
*----------------------------------------*
      ALrr=-Int0+Int0p
      ALrp=0.d0
      ALpr=0.d0
      ALpp=0.d0
      do m=1,m1
      ALrr=ALrr-cphi1(m)*((2*m+1)*Int1(m)+(2*m-1)*Int2(m))
      ALrp=ALrp-2.d0*sphi1(m)*m*(Int1(m)-Int2(m))
      ALpr=ALpr-sphi1(m)*((2*m+1)*Int1(m)-(2*m-1)*Int2(m))
      ALpp=ALpp+2.d0*cphi1(m)*m*(Int1(m)+Int2(m))
      enddo
      do m=1,m3
      ALrr=ALrr+4.d0*cphi1(m)*Int3(m)
      enddo
      ALrr=ALrr*(kappa0/r)
      ALrp=ALrp*(kappa0/r)
      ALpr=ALpr*(kappa0/r)
      ALpp=ALpp*(kappa0/r)
      PSI11=cphi**2*ALrr-cphi*sphi*ALpr+sphi**2*ALr/r
     a     -sphi*cphi*ALrp+sphi*cphi*ALphi/r+sphi**2*ALpp
      PSI22=sphi**2*ALrr+sphi*cphi*ALpr+cphi**2*ALr/r
     a     +sphi*cphi*ALrp-sphi*cphi*ALphi/r+cphi**2*ALpp
      PSI12=sphi*cphi*ALrr-sphi**2*ALpr-sphi*cphi*ALr/r
     a     +cphi**2*ALrp-cphi**2*ALphi/r-sphi*cphi*ALpp
      PSI21=PSI12
  100 dfl(1)=ALx
      dfl(2)=ALy
      mgnf(1)=1.d0-PSI11
      mgnf(2)=-PSI12
      mgnf(3)=-PSI21
      mgnf(4)=1.d0-PSI22
      zeta=MAX(abs(mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3)),1.d-6)
      zeta=DSIGN(zeta,mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3))
      mgnf(5)=1.d0/zeta 
      return
*
      END  
**********************************************************************
      FUNCTION Int2m1(mu,P,Q,r,eps)
      Implicit none
      Integer k
      Double Precision Int2m1,mu,P,Q,r,pi,sum0,sum,subsum
      Double Precision eps,dpk,gamma1
      Parameter(pi=3.14159265359d0)
      External subsum
      k=0
      sum=0.d0
      gamma1=1.d0
   10 k=k+1
      dpk=DBLE(k)
      sum0=sum
      gamma1=gamma1*(mu+dpk)
      sum=sum+(-P*r**2)**k*gamma1*subsum(k,P,Q)/dpk
      if (abs((sum-sum0)/sum) .gt. eps) go to 10
      Int2m1=sum*(-r/pi)-(P/Q)*(1.d0-SQRT(1.d0-(Q/P)**2))*r
      return
      END
**********************************************************************
      FUNCTION subsum(k,P,Q)
      Implicit none
      Integer l,k
      Double Precision subsum,P,Q,pi,gamma1,gammak
      Parameter(pi=3.14159265359d0)
      gammak=1.d0
      if (k .gt. 1) then
         do l=1,k-1
         gammak=gammak*l
         enddo
      end if
      gamma1=pi/(4.d0*gammak)
      subsum=(2.d0*Q/P)*gamma1
      l=1
 10   l=l+2
      if (l .le. k) then
         gamma1=gamma1*DBLE((k-l+2)*(k-l+1))/DBLE(4*(l**2-1))
         subsum=subsum+(2.d0*Q/P)**l*gamma1
         go to 10
      end if
      return
      END      
**********************************************************************
      FUNCTION Icos_km(k,m)
C integral of cos^k(x)*cos(mx) from x=0 to pi/2
C Uses gammln.
      Implicit none
      Integer k,m,l,i,rem4,rem2
      Double Precision Icos_km,sum,pi,gammln,dpk,dpl,dpm
      Parameter(pi=3.14159265359d0)
      External gammln
      IF (k .lt. m) THEN
        rem4=INT(mod(real(m-k),4.))
        If (rem4 .eq. 0 .or. rem4 .eq. 2) Then
          Icos_km=0.d0
          return
        Else
          sum=1.d0
          do i=1,k
          sum=sum*DBLE(i)/DBLE(m-k+2*(i-1))
          enddo
          sum=sum/DBLE(m+k)
          if (rem4 .eq. 1) then
            Icos_km=sum
            return
          else
            Icos_km=-sum
            return
          endif
         End If
       ELSE
         l=INT(REAL(k-m)/2.)
         rem2=INT(mod(real(k-m),2.))
         dpk=DBLE(k)
         dpl=DBLE(l)
         dpm=DBLE(m)
         If (rem2 .eq. 0) Then
           if (l .eq. 0) then
             Icos_km=pi/DBLE(2**(k+1))
             return
           else
             Icos_km=pi*EXP(gammln(dpk+1.d0)-gammln(dpl+1.d0)
     a               -gammln(dpk-dpl+1.d0)-(dpk+1.d0)*LOG(2.d0))
             return
           endif
         Else
           if (l .eq. 0) then
             Icos_km=EXP(DBLE(m)*LOG(2.d0)+gammln(dpk+1.d0)
     a              +gammln(dpm+1.d0)-gammln(2.d0*dpm+2.d0))
             return
           else
             Icos_km=EXP(DBLE(m+2*l)*LOG(2.d0)+gammln(dpk+1.d0)
     a              +gammln(dpl+1.d0)+gammln(dpm+dpl+1.d0)
     a     -gammln(2.d0*dpl+2.d0)-gammln(2.d0*dpl+2.d0*dpm+2.d0))
             return
           endif
         End If
      END IF
      END
**********************************************************************
      FUNCTION hg2F1(a0,b0,c0,x,eps)
      Implicit none
      Integer j,lim
      Double Precision a0,b0,c0,a,b,c,d,t,s,s0,x,eps,hg2F1
      a=a0
      b=b0
      c=c0
      d=1.d0
      t=1.d0
      s=1.d0
      if (a0 .gt. -1.d0 .and. b0 .gt. -1.d0) then
         lim=2
      else
         lim=2*(MAX(INT(-a0),INT(-b0))+1)
      endif
      do j=1,lim
      t=t*(a/c)*(b/d)*x
      if (t .eq. 0.d0) then
         hg2F1=s
         return
      end if      
      s=s+t
      a=a+1.d0
      b=b+1.d0
      c=c+1.d0
      d=d+1.d0
      enddo      
   5  s0=s
      t=t*(a/c)*(b/d)*x
      if (t .eq. 0.d0) then
         hg2F1=s
         return
      end if      
      s=s+t
      a=a+1.d0
      b=b+1.d0
      c=c+1.d0
      d=d+1.d0
      if (abs((s-s0)/s0) .gt. eps*1.d-1) go to 5
      hg2F1=s
      return
      END
**********************************************************************
      FUNCTION gammln(xx)
      DOUBLE PRECISION gammln,xx
      INTEGER j
      DOUBLE PRECISION ser,stp,tmp,x,y,cof(6)
      SAVE cof,stp
      DATA cof,stp/76.18009172947146d0,-86.50532032941677d0,
     *24.01409824083091d0,-1.231739572450155d0,.1208650973866179d-2,
     *-.5395239384953d-5,2.5066282746310005d0/
      x=xx
      y=x
      tmp=x+5.5d0
      tmp=(x+0.5d0)*log(tmp)-tmp
      ser=1.000000000190015d0
      do 11 j=1,6
        y=y+1.d0
        ser=ser+cof(j)/y
11    continue
      gammln=tmp+log(stp*ser/x)
      return
      END
C Taken from Numerical Recipes
