******************************************************************************
* This FORTRAN subroutine calculates the deflection and magnification by a   *
* family of cusped two power-law ellipsoidal mass distributions using Fourier*
* series expansion techniques.                                               *
* The projected surface mass density takes the following forms,              *
* for r > (or =) r_b/(1+e)^(1/2)                                             *
* K(z) = K_0 B[1/2,(nu_o-1)/2] (1+z^2)^[-(nu_o-1)/2]                         *
*        F[nu_i/2,(nu_o-1)/2;nu_o/2;1/(1+z^2)]                               *
* and for r < r_b/(1+e)^(1/2)                                                *
* K(z) = K_0 B[1/2,(nu_i-1)/2] z^[-(nu_i-1)] (1+z^2)^[-(nu_o-nu_i)/2]        *
*        F[1/2,(nu_o-nu_i)/2;(3-nu_i)/2;z^2/(1+z^2)] +                       *
*        K_0 B[(nu_o-1)/2,(1-nu_i)/2] (1+z^2)^[-(nu_o-1)/2]                  *
*        F[nu_i/2,(nu_o-1)/2;(nu_i+1)/2;z^2/(1+z^2)]                         *
*                                                                            *
* Here                                                                       *
*      z := (r/r_b)[1 + e cos 2(phi-phi_0)]^(1/2)                            *
*      B(a,b) : beta function                                                *
*      F(a,b;c;z) : hypergeometric function                                  *
*      K_0 (``kappa zero''): dimensionless surface density  => prmt(1)       *
*      nu_i, nu_o : inner & outer radial indices  => prmt(2) & prmt(3)       *
*      r_b : break radius  => prmt(4)                                        *
*      e (non-negative and < 1): parameter related to the ellipticity via    *
*          ellipticity = 1 - [(1 - e)/(1 + e)]^(1/2).                        *
*          ellipticity (not e) => prmt(5)                                    *
*      phi_0 : standard position angle (north through east) => prmt(6)       *
*      x [= r cos(phi)] & y [= r sin(phi)] coordinates on the lens plane     *
*        => prmt(7) & prmt(8)                                                *
*      Finally, prmt(9) is the parameter controlling the truncation of the   *
*      series.                                                               *
*   ***Important Notes***                                                    *
*      The errors for the calculated deflection and magnification components *
*      are smaller than the given value of prmt(9). In particular, for       *
*      relatively low ellipticities (ellipticity < 0.4), the errors are much *
*      smaller than prmt(9).                                                 *
*      However, for the following special cases, the above statement or the  *
*      accuracy of the calculation by the present code cannot be assured:    *
*      (1) nu_i = 1 and/or nu_o = 3,                                         *
*      (2) 0.7 < r/[r_b SQRT(1+e)] < 1.3 with ellipticity > 0.4.             *
*      Please see Chae (2002). Any improvememnts in the accuracy and speed   *
*      for these special cases will be notified at `www.jb.man.ac.uk/~chae'. *
* The outputs are as follows:                                                *
*                                                                            *
*     dfl(1) & dfl(2): x & y components of the deflection angle              *
*     mgnf(1) = 1 - psi_(,11)                                                *
*     mgnf(2) = -psi_(,12)                                                   *
*     mgnf(3) = -psi_(,21) = mgnf(2)                                         *
*     mgnf(4) = 1 - psi_(,22)                                                *
*     mgnf(5) = Magnification factor = [mgnf(1)*mgnf(4)-mgnf(2)^2]^(-1)      *
*                                                                            *
* This subroutine should be compiled together with `spleFS.f' since the      *
* latter is called by the former.                                            *
* References: 1. Chae K.-H. 2002, ApJ, in press                              *
*             2. Chae K.-H., Khersonsky V.K., Turnshek D.A. 1998, ApJ, 506,80*
* Last modified on 8th December 2001, Kyu-Hyun Chae                          *
* Please send your comments to chae@jb.man.ac.uk or chae@phyast.pitt.edu     *
******************************************************************************
      SUBROUTINE cuspFS(prmt,dfl,mgnf)
* cuspFS stands for cusp model Fourier Series.
* Uses cuspFS_1, cuspFS_2. 
      Implicit none
      Integer i
      Double Precision prmt(9),dfl(2),mgnf(5)
      Double Precision nu_in,nu_out,rb,ellip,e,x,y,r,ra,delnu
      Double Precision dfl1(2),mgnf1(5),dfl2(2),mgnf2(5)
      Double Precision dfl3(2),mgnf3(5),dfl4(2),mgnf4(5),zeta,cof(4)
      nu_in=prmt(2)
      nu_out=prmt(3)
      rb=prmt(4)
      ellip=prmt(5)
      x=prmt(7)
      y=prmt(8)
      e=ABS(ellip*(2.d0-ellip)/(1.d0+(1.d0-ellip)**2))
      if (abs(x/rb) .lt. 1.d-10 .and. abs(y/rb) .lt. 1.d-10) then
         r=1.d-10*rb
      else
         r=sqrt(x**2+y**2)
      end if
      ra=rb*0.8d0/SQRT(1.d0+e)
      if (r .ge. rb*0.6d0/SQRT(1.d0+e) .and. 
     a    r .lt. rb*1.3d0/SQRT(1.d0+e)) then
         delnu=8.d-3
      else
         delnu=2.d-3
      endif
*
      IF (r .lt. ra) THEN
      If (ABS(nu_in-1.d0) .ge. delnu/2.d0 .and. 
     a    ABS(nu_out-3.d0) .ge. delnu/2.d0) Then
      call cuspFS_1(prmt,dfl,mgnf)
      return
* Below treat the special cases nu_in = 1 and/or nu_out = 3 using
* interpolation. 
      Else If (ABS(nu_in-1.d0) .lt. delnu/2.d0 .and. 
     a         ABS(nu_out-3.d0) .ge. delnu/2.d0) Then
      prmt(2)=1.d0-delnu/2.d0
      call cuspFS_1(prmt,dfl1,mgnf1)
      prmt(2)=1.d0+delnu/2.d0
      call cuspFS_1(prmt,dfl2,mgnf2)
      dfl(1)=dfl1(1)+((dfl2(1)-dfl1(1))/delnu)*(nu_in-1.d0+delnu/2.d0)
      dfl(2)=dfl1(2)+((dfl2(2)-dfl1(2))/delnu)*(nu_in-1.d0+delnu/2.d0)
      do i=1,4
      mgnf(i)=mgnf1(i)
     a        +((mgnf2(i)-mgnf1(i))/delnu)*(nu_in-1.d0+delnu/2.d0)
      enddo
      zeta=MAX(abs(mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3)),1.d-6)
      zeta=DSIGN(zeta,mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3))
      mgnf(5)=1.d0/zeta 
      return         
      Else If (ABS(nu_in-1.d0) .ge. delnu/2.d0 .and. 
     a         ABS(nu_out-3.d0) .lt. delnu/2.d0) Then
      prmt(3)=3.d0-delnu/2.d0
      call cuspFS_1(prmt,dfl1,mgnf1)
      prmt(3)=3.d0+delnu/2.d0
      call cuspFS_1(prmt,dfl2,mgnf2)
      dfl(1)=dfl1(1)+((dfl2(1)-dfl1(1))/delnu)*(nu_out-3.d0+delnu/2.d0)
      dfl(2)=dfl1(2)+((dfl2(2)-dfl1(2))/delnu)*(nu_out-3.d0+delnu/2.d0)
      do i=1,4
      mgnf(i)=mgnf1(i)
     a        +((mgnf2(i)-mgnf1(i))/delnu)*(nu_out-3.d0+delnu/2.d0)
      enddo
      zeta=MAX(abs(mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3)),1.d-6)
      zeta=DSIGN(zeta,mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3))
      mgnf(5)=1.d0/zeta 
      return         
      Else
      prmt(2)=1.d0-delnu/2.d0
      prmt(3)=3.d0-delnu/2.d0
      call cuspFS_1(prmt,dfl1,mgnf1)
      prmt(2)=1.d0+delnu/2.d0
      prmt(3)=3.d0-delnu/2.d0
      call cuspFS_1(prmt,dfl2,mgnf2)
      prmt(2)=1.d0-delnu/2.d0
      prmt(3)=3.d0+delnu/2.d0
      call cuspFS_1(prmt,dfl3,mgnf3)
      prmt(2)=1.d0+delnu/2.d0
      prmt(3)=3.d0+delnu/2.d0
      call cuspFS_1(prmt,dfl4,mgnf4)
      cof(1)=1.d0-(nu_in-1.d0+delnu/2.d0)/delnu
     a      +((nu_out-3.d0+delnu/2.d0)/delnu)
     a       *(-1.d0+(nu_in-1.d0+delnu/2.d0)/delnu) 
      cof(2)=((nu_in-1.d0+delnu/2.d0)/delnu)
     a       *(1.d0-(nu_out-3.d0+delnu/2.d0)/delnu)
      cof(3)=((nu_out-3.d0+delnu/2.d0)/delnu)
     a       *(1.d0-(nu_in-1.d0+delnu/2.d0)/delnu)
      cof(4)=((nu_in-1.d0+delnu/2.d0)/delnu)
     a       *((nu_out-3.d0+delnu/2.d0)/delnu)
      dfl(1)=cof(1)*dfl1(1)+cof(2)*dfl2(1)+cof(3)*dfl3(1)+cof(4)*dfl4(1)
      dfl(2)=cof(1)*dfl1(2)+cof(2)*dfl2(2)+cof(3)*dfl3(2)+cof(4)*dfl4(2)
      do i=1,4
      mgnf(i)=cof(1)*mgnf1(i)+cof(2)*mgnf2(i)
     a       +cof(3)*mgnf3(i)+cof(4)*mgnf4(i)
      enddo
      zeta=MAX(abs(mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3)),1.d-6)
      zeta=DSIGN(zeta,mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3))
      mgnf(5)=1.d0/zeta 
      return
      End If
*
      ELSE
      If (ABS(nu_in-1.d0) .ge. delnu/2.d0 .and. 
     a    ABS(nu_out-3.d0) .ge. delnu/2.d0) Then
      call cuspFS_2(prmt,dfl,mgnf)
      return
* Below treat the special cases nu_in = 1 and/or nu_out = 3 using
* interpolation. 
      Else If (ABS(nu_in-1.d0) .lt. delnu/2.d0 .and. 
     a         ABS(nu_out-3.d0) .ge. delnu/2.d0) Then
      prmt(2)=1.d0-delnu/2.d0
      call cuspFS_2(prmt,dfl1,mgnf1)
      prmt(2)=1.d0+delnu/2.d0
      call cuspFS_2(prmt,dfl2,mgnf2)
      dfl(1)=dfl1(1)+((dfl2(1)-dfl1(1))/delnu)*(nu_in-1.d0+delnu/2.d0)
      dfl(2)=dfl1(2)+((dfl2(2)-dfl1(2))/delnu)*(nu_in-1.d0+delnu/2.d0)
      do i=1,4
      mgnf(i)=mgnf1(i)
     a        +((mgnf2(i)-mgnf1(i))/delnu)*(nu_in-1.d0+delnu/2.d0)
      enddo
      zeta=MAX(abs(mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3)),1.d-6)
      zeta=DSIGN(zeta,mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3))
      mgnf(5)=1.d0/zeta 
      return         
      Else If (ABS(nu_in-1.d0) .ge. delnu/2.d0 .and. 
     a         ABS(nu_out-3.d0) .lt. delnu/2.d0) Then
      prmt(3)=3.d0-delnu/2.d0
      call cuspFS_2(prmt,dfl1,mgnf1)
      prmt(3)=3.d0+delnu/2.d0
      call cuspFS_2(prmt,dfl2,mgnf2)
      dfl(1)=dfl1(1)+((dfl2(1)-dfl1(1))/delnu)*(nu_out-3.d0+delnu/2.d0)
      dfl(2)=dfl1(2)+((dfl2(2)-dfl1(2))/delnu)*(nu_out-3.d0+delnu/2.d0)
      do i=1,4
      mgnf(i)=mgnf1(i)
     a        +((mgnf2(i)-mgnf1(i))/delnu)*(nu_out-3.d0+delnu/2.d0)
      enddo
      zeta=MAX(abs(mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3)),1.d-6)
      zeta=DSIGN(zeta,mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3))
      mgnf(5)=1.d0/zeta 
      return         
      Else
      prmt(2)=1.d0-delnu/2.d0
      prmt(3)=3.d0-delnu/2.d0
      call cuspFS_2(prmt,dfl1,mgnf1)
      prmt(2)=1.d0+delnu/2.d0
      prmt(3)=3.d0-delnu/2.d0
      call cuspFS_2(prmt,dfl2,mgnf2)
      prmt(2)=1.d0-delnu/2.d0
      prmt(3)=3.d0+delnu/2.d0
      call cuspFS_2(prmt,dfl3,mgnf3)
      prmt(2)=1.d0+delnu/2.d0
      prmt(3)=3.d0+delnu/2.d0
      call cuspFS_2(prmt,dfl4,mgnf4)
      cof(1)=1.d0-(nu_in-1.d0+delnu/2.d0)/delnu
     a      +((nu_out-3.d0+delnu/2.d0)/delnu)
     a       *(-1.d0+(nu_in-1.d0+delnu/2.d0)/delnu) 
      cof(2)=((nu_in-1.d0+delnu/2.d0)/delnu)
     a       *(1.d0-(nu_out-3.d0+delnu/2.d0)/delnu)
      cof(3)=((nu_out-3.d0+delnu/2.d0)/delnu)
     a       *(1.d0-(nu_in-1.d0+delnu/2.d0)/delnu)
      cof(4)=((nu_in-1.d0+delnu/2.d0)/delnu)
     a       *((nu_out-3.d0+delnu/2.d0)/delnu)
      dfl(1)=cof(1)*dfl1(1)+cof(2)*dfl2(1)+cof(3)*dfl3(1)+cof(4)*dfl4(1)
      dfl(2)=cof(1)*dfl1(2)+cof(2)*dfl2(2)+cof(3)*dfl3(2)+cof(4)*dfl4(2)
      do i=1,4
      mgnf(i)=cof(1)*mgnf1(i)+cof(2)*mgnf2(i)
     a       +cof(3)*mgnf3(i)+cof(4)*mgnf4(i)
      enddo
      zeta=MAX(abs(mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3)),1.d-6)
      zeta=DSIGN(zeta,mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3))
      mgnf(5)=1.d0/zeta 
      return
      End If
      END IF
*
      END
******************************************************************************
* cuspFS for r < 0.8 r_b/(1+e)^(1/2)                                         *
******************************************************************************
      SUBROUTINE cuspFS_1(prmt,dfl,mgnf)
* Uses spleFS, cuspFS_1a, gammln, hg2F1. 
      Implicit none
      Integer n,nmax,l,i
      Parameter(nmax=200)
      Double Precision prmt(9),dfl(2),mgnf(5)
      Double Precision kappa0,nu_in,nu_out,rb,ellip,phi0,x,y
      Double Precision r,phi,eps,e,pi,lam,mu,tr,gammln,C1,C2,C3
      Double Precision hg2F1,sum_dflx,sum0_dflx,sum_dfly,sum0_dfly
      Double Precision sum_psixx,sum_psixy,sum_psiyy,nu,ALx,ALy
      Double Precision ALr,ALrr,PSI11,PSI12,PSI21,PSI22,cphi,sphi
      Double Precision prmt_spleFS(8),kappa0_spleFS,dfl1(2),mgnf1(5)
      Double Precision dfl2(2),mgnf2(5),zeta,sum,sum0
      Double Precision defmag(0:nmax,1:5),t(5),term0,term
      Double Precision rat,zeroth,dpn,dpl
      Parameter(pi=3.14159265359d0)
      External gammln,hg2F1
      kappa0=prmt(1)
      nu_in=prmt(2)
      nu_out=prmt(3)
      rb=prmt(4)
      ellip=prmt(5)
      phi0=prmt(6)
      x=prmt(7)
      y=prmt(8)
      eps=prmt(9)
      do n=3,8
      prmt_spleFS(n)=prmt(n+1)
      enddo
      e=ABS(ellip*(2.d0-ellip)/(1.d0+(1.d0-ellip)**2))
      if (abs(x) .lt. 1.d-10 .and. abs(y) .lt. 1.d-10) then
         r=0.d0
      else
         r=sqrt(x**2+y**2)
      end if
      phi=datan2(y,x)
*
      If (e .lt. 1.d-6) then
         tr=(r/rb)**2
         cphi=cos(phi)
         sphi=sin(phi)
         C2=(nu_in*SQRT(pi)/(nu_in-1.d0))
     a    *EXP(gammln((nu_in+1.d0)/2.d0)-gammln((nu_in+2.d0)/2.d0))
         lam=(-nu_in+1.d0)/2.d0
         mu=(nu_out-nu_in)/2.d0-1.d0
         ALr=(kappa0*C2*rb**2/((lam+1.d0)*r))
     a       *(tr**(lam+1.d0)/(1.d0+tr)**(mu+1.d0))
     a       *hg2F1(mu+1.d0,1.d0,lam+2.d0,tr/(1.d0+tr),eps)
         ALx=ALr*cphi
         ALy=ALr*sphi
         ALrr=-ALr/r+2.d0*kappa0*C2*tr**lam/(1.d0+tr)**(mu+1.d0)
         PSI11=cphi**2*ALrr+sphi**2*ALr/r
         PSI22=sphi**2*ALrr+cphi**2*ALr/r
         PSI12=sphi*cphi*ALrr-sphi*cphi*ALr/r
         PSI21=PSI12          
         dfl1(1)=ALx
         dfl1(2)=ALy
         mgnf1(1)=1.d0-PSI11
         mgnf1(2)=-PSI12
         mgnf1(3)=-PSI21
         mgnf1(4)=1.d0-PSI22
         sum_dflx=dfl1(1)
         sum_dfly=dfl1(2)
         sum=sum_dflx**2+sum_dfly**2
         sum_psixx=1.d0-mgnf1(1)
         sum_psixy=-mgnf1(2)
         sum_psiyy=1.d0-mgnf1(4)
         n=0
 11      n=n+1
         dpn=DBLE(n)
         sum0_dflx=sum_dflx
         sum0_dfly=sum_dfly
         sum0=sum
         C2=C2*(dpn+0.5d0*(nu_out-nu_in)-1.d0)*(dpn-0.5d0)
     a        /((dpn+0.5d0*(1.d0-nu_in))*dpn)
         lam=lam+1.d0
         mu=mu+1.d0
         ALr=(kappa0*C2*rb**2/((lam+1.d0)*r))
     a       *(tr**(lam+1.d0)/(1.d0+tr)**(mu+1.d0))
     a       *hg2F1(mu+1.d0,1.d0,lam+2.d0,tr/(1.d0+tr),eps)
         ALx=ALr*cphi
         ALy=ALr*sphi
         ALrr=-ALr/r+2.d0*kappa0*C2*tr**lam/(1.d0+tr)**(mu+1.d0)
         PSI11=cphi**2*ALrr+sphi**2*ALr/r
         PSI22=sphi**2*ALrr+cphi**2*ALr/r
         PSI12=sphi*cphi*ALrr-sphi*cphi*ALr/r
         PSI21=PSI12          
         dfl1(1)=ALx
         dfl1(2)=ALy
         mgnf1(1)=1.d0-PSI11
         mgnf1(2)=-PSI12
         mgnf1(3)=-PSI21
         mgnf1(4)=1.d0-PSI22
         sum_dflx=sum_dflx+dfl1(1)
         sum_dfly=sum_dfly+dfl1(2)
         sum_psixx=sum_psixx+1.d0-mgnf1(1)
         sum_psixy=sum_psixy-mgnf1(2)
         sum_psiyy=sum_psiyy+1.d0-mgnf1(4)
         sum=sum_dflx**2+sum_dfly**2
         if (ABS(0.5d0*(sum-sum0)/sum0) .gt. eps) go to 11
         sum_dflx=sum_dflx-dfl1(1)
         sum_dfly=sum_dfly-dfl1(2)
         sum_psixx=sum_psixx-(1.d0-mgnf1(1))
         sum_psixy=sum_psixy-(-mgnf1(2))
         sum_psiyy=sum_psiyy-(1.d0-mgnf1(4))
         dfl(1)=sum_dflx
         dfl(2)=sum_dfly
         mgnf(1)=1.d0-sum_psixx
         mgnf(2)=-sum_psixy
         mgnf(3)=-sum_psixy
         mgnf(4)=1.d0-sum_psiyy
      Else
         call cuspFS_1a(prmt,dfl1,mgnf1)
         dfl(1)=dfl1(1)
         dfl(2)=dfl1(2)
         mgnf(1)=mgnf1(1)
         mgnf(2)=mgnf1(2)
         mgnf(3)=mgnf1(3)
         mgnf(4)=mgnf1(4)
      End If
*
      C3=(4.d0*(nu_out-nu_in)
     a    /((nu_out-1.d0)*(1.d0-nu_in)*(3.d0-nu_in)))
     a   *EXP(gammln((nu_out+1.d0)/2.d0)+gammln((5.d0-nu_in)/2.d0)
     a        -gammln((nu_out-nu_in+2.d0)/2.d0))
      kappa0_spleFS=kappa0*C3
      nu=nu_out
      prmt_spleFS(1)=1.d0
      prmt_spleFS(2)=nu
      prmt_spleFS(8)=eps
      call spleFS(prmt_spleFS,dfl2,mgnf2)
      sum_dflx=kappa0_spleFS*dfl2(1)
      sum_dfly=kappa0_spleFS*dfl2(2)
      sum=sum_dflx**2+sum_dfly**2
      sum_psixx=kappa0_spleFS*(1.d0-mgnf2(1))
      sum_psixy=-kappa0_spleFS*mgnf2(2)
      sum_psiyy=kappa0_spleFS*(1.d0-mgnf2(4))
      zeroth=sum
      rat=1.d0
      n=0
      defmag(n,1)=dfl2(1)
      defmag(n,2)=dfl2(2)
      defmag(n,3)=1.d0-mgnf2(1)
      defmag(n,4)=-mgnf2(2)
      defmag(n,5)=1.d0-mgnf2(4)
 12   n=n+1
      dpn=DBLE(n)
      if (n .gt. 1) term0=term
      sum0_dflx=sum_dflx
      sum0_dfly=sum_dfly
      sum0=sum
      prmt_spleFS(2)=2.d0*dpn+nu_out
      prmt_spleFS(8)=eps
      call spleFS(prmt_spleFS,dfl2,mgnf2)
      defmag(n,1)=dfl2(1)
      defmag(n,2)=dfl2(2)
      defmag(n,3)=1.d0-mgnf2(1)
      defmag(n,4)=-mgnf2(2)
      defmag(n,5)=1.d0-mgnf2(4)
      C1=1.d0
      do i=1,5
      t(i)=defmag(0,i)
      enddo
      do l=1,n
      dpl=DBLE(l)
      C1=-C1*(dpn/dpl-1.d0+1.d0/dpl)
      do i=1,5      
      t(i)=t(i)+C1*defmag(l,i)
      enddo
      enddo
      C3=C3*(dpn+0.5d0*nu_in-1.d0)*(dpn+0.5d0*nu_out-1.5d0)
     a     /((dpn+0.5d0*nu_in-0.5d0)*dpn)
      kappa0_spleFS=kappa0*C3
      sum_dflx=sum_dflx+kappa0_spleFS*t(1)
      sum_dfly=sum_dfly+kappa0_spleFS*t(2)
      sum_psixx=sum_psixx+kappa0_spleFS*t(3)
      sum_psixy=sum_psixy+kappa0_spleFS*t(4)
      sum_psiyy=sum_psiyy+kappa0_spleFS*t(5)
      sum=sum_dflx**2+sum_dfly**2
      term=ABS(sum-sum0)
      rat=ABS(zeroth/((kappa0_spleFS*t(1))**2+(kappa0_spleFS*t(2))**2))
      if (n .le. 2) then
         go to 12
      else
         if (term .lt. term0) then
         if (ABS(0.5d0*(sum-sum0)/sum0) .gt. eps) go to 12
         else
         sum_dflx=sum_dflx-kappa0_spleFS*t(1)
         sum_dfly=sum_dfly-kappa0_spleFS*t(2)
         sum_psixx=sum_psixx-kappa0_spleFS*t(3)
         sum_psixy=sum_psixy-kappa0_spleFS*t(4)
         sum_psiyy=sum_psiyy-kappa0_spleFS*t(5)
         endif
      endif
      dfl(1)=dfl(1)+sum_dflx
      dfl(2)=dfl(2)+sum_dfly
      mgnf(1)=mgnf(1)-sum_psixx
      mgnf(2)=mgnf(2)-sum_psixy
      mgnf(3)=mgnf(3)-sum_psixy
      mgnf(4)=mgnf(4)-sum_psiyy
      zeta=MAX(abs(mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3)),1.d-6)
      zeta=DSIGN(zeta,mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3))
      mgnf(5)=1.d0/zeta          
      return
*
      END  
******************************************************************************
* cuspFS 1st series for r < 0.8 r_b/(1+e)^(1/2)                              *
******************************************************************************
      SUBROUTINE cuspFS_1a(prmt,dfl,mgnf)
* Uses hg2F1,gammln.
* "I-functions" are denoted by the notations "Int*" where Int means Integral.
      Implicit none
      Integer m,m1,mmax,k,klim,kmax_B1,l,n
      Parameter(n=200)
      Double Precision prmt(9),dfl(2),mgnf(5)
      Double Precision kappa0,nu_in,nu_out,lam,mu,rb,ellip,e
      Double Precision phi0,eps,pi,r,phi,phi1,x,y,f,e1,e1sqrt,e2
      Double Precision Int0,Int0p,Int1(0:n),Int2(0:n),Int3(0:n)
      Double Precision hg2F1,Fval_lamk0(0:n),Fval_lamk0a(0:n)
      Double Precision hgt1,hgt2,hgt3,a,b,c,dpm,dpk,dpl,fac
      Double Precision sum0,sum,sum0_1,sum_1,sum0_2,sum_2
      Double Precision sum0_3,sum_3,sum0_int1,sum_int1
      Double Precision sum0_int2,sum_int2,sum0_int3,sum_int3
      Double Precision gammln,C1,C2,C3,C1k,C2k,C3k,C1p,C2p,C3p
      Double Precision B1,B1val(0:n),t0,t,t1,t2,t3,fracmax
      Double Precision ALr,ALphi,ALx,ALy,ALrr,ALrp,ALpr,ALpp
      Double Precision PSI11,PSI12,PSI21,PSI22,cphi,sphi,zeta 
      Double Precision cphi1(n),sphi1(n)
      Parameter(pi=3.14159265359d0)
      External hg2F1,gammln
*
      kappa0=prmt(1)
      nu_in=prmt(2)
      nu_out=prmt(3)
      rb=prmt(4)
      ellip=MIN(0.95d0,abs(prmt(5)))
      phi0=prmt(6)
      x=prmt(7)
      y=prmt(8)
      eps=prmt(9)
      lam=-(nu_in-1.d0)/2.d0
      mu=(nu_out-nu_in-2.d0)/2.d0
      e=ABS(ellip*(2.d0-ellip)/(1.d0+(1.d0-ellip)**2))
      if (abs(x/rb) .lt. 1.d-10 .and. abs(y/rb) .lt. 1.d-10) then
         r=1.d-10*rb
      else
         r=sqrt(x**2+y**2)
      end if
      phi=datan2(y,x)
*
* lensing by an elliptical mass distribution
*-----------------------------------------------    
* r/r_b < 0.8/(1 + e)^(1/2) & lambda=non-integer
*-----------------------------------------------
      f=SQRT(1.d0-e**2)
      e1=(1.d0-f)/(1.d0+f)
      e1sqrt=sqrt(e1)
      e2=0.5d0*(1.d0+f)
      t=(r/rb)**2*e2
      if (r/rb .lt. 2.d-10) then
        klim=1
      else
        klim=ABS((LOG(eps)-2.d0*LOG(10.d0))/LOG(t))
      endif
*
*-------------------------------------------------------------*
* Evaluation of F(m-lam,-lam;m+1;e1) & F(m-1-lam,-lam;m+1;e1) *
* for m = 0 to mmax (initial estimate).                       *
*-------------------------------------------------------------*
* Determine an initial estimate of mmax:
      C1=1.d0
      sum=1.d0
      m=0
 3    sum0=sum
      m=m+1
      dpm=DBLE(m)
      C1=C1*(-lam+dpm-1.d0)/dpm
      sum=sum+C1*e1sqrt**m
      if (ABS((sum-sum0)/sum) .gt. eps) go to 3
      mmax=m+INT(0.3*m)
*
      dpm=DBLE(mmax)
      hgt1=hg2F1(dpm+1.d0-lam,-lam,dpm+2.d0,e1,eps)
      hgt2=hg2F1(dpm-lam,-lam,dpm+2.d0,e1,eps)
      Fval_lamk0(mmax+1)=hgt1
      Fval_lamk0a(mmax+1)=hgt2
      a=dpm+1.d0-lam
      b=-lam
      c=dpm+2.d0
      do m=mmax,0,-1
      hgt3=hgt1*(a-1.d0)/(c-1.d0)+hgt2*(c-a)/(c-1.d0)
      hgt2=hgt2*e1*(c-b-1.d0)/(c-1.d0)+(1.d0-e1)*hgt3
      hgt1=hgt3
      a=a-1.d0
      c=c-1.d0
      Fval_lamk0(m)=hgt3
      Fval_lamk0a(m)=hgt2
      enddo
      mmax=mmax+1
*
*------------------------------*
* Evaluation of Int0 and Int0p *
*------------------------------*
      hgt1=-Fval_lamk0a(0)*(1.d0/lam+1.d0)/(1.d0-e1)
     a     +Fval_lamk0(0)*(1.d0/lam+2.d0)/(1.d0-e1)
      hgt2=Fval_lamk0(0)
      a=-lam+1.d0
      b=-lam
      c=1.d0
      C1=1.d0/(lam+1.d0)
      C3=1.d0
      C1k=C1
      C3k=C3
      B1=(nu_in*SQRT(pi)/(nu_in-1.d0))
     a   *EXP(gammln((nu_in+1.d0)/2.d0)-gammln((nu_in+2.d0)/2.d0))
      B1val(0)=B1
      sum_1=B1*C1*hgt2
      sum_3=B1*C3*hgt2
      t0=1.d0
      k=0
 10   sum0_1=sum_1
      sum0_3=sum_3
      k=k+1
      dpk=DBLE(k)
      hgt3=(a-1.d0)/(c-b)*(1.d0-e1)*hgt1
     a    +((b-1.d0)*(c-b)-(a-1.d0)*(c-a))/((b-a)*(c-b))*hgt2
      hgt2=(c+2.d0-2.d0*a-(b-a)*e1)/(c-a+1.d0)*hgt3
     a    -((a-1.d0)*(b-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)**2*hgt1
     a    +((a-1.d0)*(c-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)*hgt2
      hgt1=hgt3
      a=a-1.d0
      b=b-1.d0
      C1k=C1k*(lam+dpk)/(lam+dpk+1.d0)
      t0=t0*t
      B1=B1*(dpk+0.5d0*(nu_out-nu_in)-1.d0)*(dpk-0.5d0)
     a     /((dpk+0.5d0*(1.d0-nu_in))*dpk)
      B1val(k)=B1
      C1p=C1k
      C3p=C3k
      t1=B1val(k)*C1p
      t3=B1val(k)*C3p
      do l=k-1,0,-1
        dpl=DBLE(l)
        fac=(mu+dpl+1.d0)/(dpk-dpl)
        C1p=-C1p*fac
        C3p=-C3p*fac
        t1=t1+B1val(l)*C1p
        t3=t3+B1val(l)*C3p
      enddo
      sum_1=sum_1+t1*t0*hgt2
      sum_3=sum_3+t3*t0*hgt2
      fracmax=MAX(ABS((sum_1-sum0_1)/sum_1)
     a           ,ABS((sum_3-sum0_3)/sum_3))
      if (k .le. klim) then
      if (fracmax .gt. eps) go to 10
      endif
      kmax_B1=k
      Int0=r*t**lam*sum_1
      Int0p=2.d0*r*t**lam*sum_3
      Int1(0)=Int0
      Int2(0)=-Int0
      Int3(0)=Int0p/2.d0
*
*------------------------------------------------------------------------*
* Evaluation of Int1(m), Int2(m) and Int3(m): keep terms for m=1 to mmax *
*------------------------------------------------------------------------*
      m=0
      sum_int1=ABS(Int1(0))
      sum_int2=ABS(Int2(0))
      sum_int3=ABS(Int3(0))
      C1=1.d0/(lam+1.d0)
      C2=-C1
      C3=1.d0
 13   sum0_int1=sum_int1
      sum0_int2=sum_int2
      sum0_int3=sum_int3
      m=m+1
      dpm=DBLE(m)
      C1=C1*((dpm-lam-1.d0)*(dpm+lam))/((dpm+lam+1.d0)*dpm)
      C2=C2*((dpm-lam-2.d0)/dpm)
      C3=C3*((dpm-lam-1.d0)/dpm)
      if (m .le. mmax) then
         hgt1=Fval_lamk0(m)
         hgt2=Fval_lamk0a(m)
      else
         hgt1=hg2F1(dpm-lam,-lam,dpm+1.d0,e1,eps)
         hgt2=hg2F1(dpm-1.d0-lam,-lam,dpm+1.d0,e1,eps)
         mmax=m
      end if
      a=dpm-lam
      b=-lam
      c=dpm+1.d0
      sum_1=B1val(0)*C1*hgt1
      sum_2=B1val(0)*C2*hgt1
      sum_3=B1val(0)*C3*hgt1
      C1k=C1
      C2k=C2
      C3k=C3
      t0=1.d0
      k=0
 14   sum0_1=sum_1
      sum0_2=sum_2
      sum0_3=sum_3
      k=k+1
      dpk=DBLE(k)
      C1k=C1k*((lam+dpm+dpk)*(dpk+lam))
     a       /((lam+dpm+dpk+1.d0)*(dpk+lam-dpm))
      C2k=C2k*(dpk+lam)/(lam-dpm+dpk+1.d0)
      C3k=C3k*(dpk+lam)/(dpk+lam-dpm)
      t0=t0*t
      if (k .eq. kmax_B1+1) then
        B1=B1val(kmax_B1)
        B1=B1*(dpk+0.5d0*(nu_out-nu_in)-1.d0)*(dpk-0.5d0)
     a       /((dpk+0.5d0*(1.d0-nu_in))*dpk)
        B1val(k)=B1
        kmax_B1=k
      endif
      hgt3=(a-1.d0)/(c-b)*(1.d0-e1)*hgt1
     a    +((b-1.d0)*(c-b)-(a-1.d0)*(c-a))/((b-a)*(c-b))*hgt2
      hgt2=(c+2.d0-2.d0*a-(b-a)*e1)/(c-a+1.d0)*hgt3
     a    -((a-1.d0)*(b-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)**2*hgt1
     a    +((a-1.d0)*(c-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)*hgt2
      hgt1=hgt3
      a=a-1.d0
      b=b-1.d0
      C1p=C1k
      C2p=C2k
      C3p=C3k
      t1=B1val(k)*C1p
      t2=B1val(k)*C2p
      t3=B1val(k)*C3p
      do l=k-1,0,-1
        dpl=DBLE(l)
        fac=(mu+dpl+1.d0)/(dpk-dpl)
        C1p=-C1p*fac
        C2p=-C2p*fac
        C3p=-C3p*fac
        t1=t1+B1val(l)*C1p
        t2=t2+B1val(l)*C2p
        t3=t3+B1val(l)*C3p
      enddo
      sum_1=sum_1+t1*t0*hgt3
      sum_2=sum_2+t2*t0*hgt3
      sum_3=sum_3+t3*t0*hgt3
      fracmax=MAX(ABS((sum_1-sum0_1)/sum0_1),
     a            ABS((sum_2-sum0_2)/sum0_2))
      fracmax=MAX(fracmax,ABS((sum_3-sum0_3)/sum0_3))
      if (k .lt. klim) then
      if (fracmax .gt. eps) go to 14
      endif
      t1=r*(-e1sqrt)**m*t**lam
      Int1(m)=t1*sum_1
      Int2(m)=t1*sum_2
      Int3(m)=t1*sum_3
      sum_int1=sum_int1+ABS(dpm*Int1(m))
      sum_int2=sum_int2+ABS(dpm*Int2(m))
      sum_int3=sum_int3+ABS(dpm*Int3(m))
      fracmax=MAX(ABS((sum_int1-sum0_int1)/sum0_int1),
     a            ABS((sum_int2-sum0_int2)/sum0_int2))
      fracmax=MAX(fracmax,ABS((sum_int3-sum0_int3)/sum0_int3))
      if (fracmax .gt. eps) go to 13
      m1=m
*-----------------------------------------------*
*Calculation of the Deflection Angle Components *
*-----------------------------------------------*
      phi1=2.d0*(phi-phi0)
      cphi=cos(phi)
      sphi=sin(phi)
      cphi1(1)=cos(phi1)
      sphi1(1)=sin(phi1)
      if (m1 .gt. 1) then
      do m=2,m1
      cphi1(m)=cphi1(1)*cphi1(m-1)-sphi1(1)*sphi1(m-1)
      sphi1(m)=sphi1(1)*cphi1(m-1)+cphi1(1)*sphi1(m-1)
      enddo
      end if
*:Alpha(r) and Alpha(phi)
      ALr=Int0
      ALphi=0.d0
      do m=1,m1
      ALr=ALr+cphi1(m)*(Int1(m)-Int2(m))
      ALphi=Alphi+sphi1(m)*(Int1(m)+Int2(m))
      enddo
      ALr=kappa0*ALr
      ALphi=kappa0*ALphi
*:transformation to Alpha(x) and Alpha(y)
      ALx=ALr*cphi-ALphi*sphi
      ALy=ALr*sphi+ALphi*cphi
*----------------------------------------*
*Calculation of the magnification matrix *
*----------------------------------------*
      ALrr=-Int0+Int0p
      ALrp=0.d0
      ALpr=0.d0
      ALpp=0.d0
      do m=1,m1
      ALrr=ALrr-cphi1(m)*((2*m+1)*Int1(m)+(2*m-1)*Int2(m))
      ALrp=ALrp-2.d0*sphi1(m)*m*(Int1(m)-Int2(m))
      ALpr=ALpr-sphi1(m)*((2*m+1)*Int1(m)-(2*m-1)*Int2(m))
      ALpp=ALpp+2.d0*cphi1(m)*m*(Int1(m)+Int2(m))
      enddo
      do m=1,m1
      ALrr=ALrr+4.d0*cphi1(m)*Int3(m)
      enddo
      ALrr=ALrr*(kappa0/r)
      ALrp=ALrp*(kappa0/r)
      ALpr=ALpr*(kappa0/r)
      ALpp=ALpp*(kappa0/r)
      PSI11=cphi**2*ALrr-cphi*sphi*ALpr+sphi**2*ALr/r
     a     -sphi*cphi*ALrp+sphi*cphi*ALphi/r+sphi**2*ALpp
      PSI22=sphi**2*ALrr+sphi*cphi*ALpr+cphi**2*ALr/r
     a     +sphi*cphi*ALrp-sphi*cphi*ALphi/r+cphi**2*ALpp
      PSI12=sphi*cphi*ALrr-sphi**2*ALpr-sphi*cphi*ALr/r
     a     +cphi**2*ALrp-cphi**2*ALphi/r-sphi*cphi*ALpp
      PSI21=PSI12
      dfl(1)=ALx
      dfl(2)=ALy
      mgnf(1)=1.d0-PSI11
      mgnf(2)=-PSI12
      mgnf(3)=-PSI21
      mgnf(4)=1.d0-PSI22
      zeta=MAX(abs(mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3)),1.d-6)
      zeta=DSIGN(zeta,mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3))
      mgnf(5)=1.d0/zeta          
      return
*
      END  
******************************************************************************
* cuspFS for r > (or =) 0.8 r_b/(1+e)^(1/2)                                  *
******************************************************************************
      SUBROUTINE cuspFS_2(prmt,dfl,mgnf)
* Uses hg2F1, gammln, Icos_km, ressum.
* "I-functions" are denoted by the notations "Int*" where Int means Integral.
      Implicit none
      Integer m,m1,m2,m3,mmax,k,klim,l,n,cmax
      Parameter(n=200)
      Double Precision prmt(9),dfl(2),mgnf(5)
      Double Precision kappa0,nu_in,nu_out,rb,mu,muinv
      Double Precision ellip,e,phi0,P,Q,eps,pi,r,phi,phi1
      Double Precision x,y,h,e1,e1sqrt,e2,e2mu,f1,f2,et,etsq
      Double Precision Int0,Int0p,Int1(0:n),Int2(0:n),Int3(0:n)
      Double Precision hg2F1,Fval_muk0(0:1000),Fval_muk0a(0:1000)
      Double Precision hgt1,hgt2,hgt3,a,b,c,hgt1p,hgt2p,hgt3p
      Double Precision ap,bp,cp,t,t1,t2,t3,sum,sum0
      Double Precision sum0_1,sum_1,sum0_11,sum_11,sum0_12,sum_12
      Double Precision sum0_2,sum_2,sum0_3,sum_3,sum0_int1,sum_int1
      Double Precision sum0_int2,sum_int2,sum0_int3,sum_int3
      Double Precision CC,CC0,C1,C2,C3,C4,C5,C6,C7,C8,C9,Cval(0:1000)
      Double Precision D1,D2,fracmax,dpm,dpk,dpl,gammln
      Double Precision ALr,ALphi,ALx,ALy,ALrr,ALrp,ALpr,ALpp
      Double Precision PSI11,PSI12,PSI21,PSI22,cphi,sphi,zeta 
      Double Precision cphi1(n),sphi1(n),term(0:1000)
      Double Precision Icos_km,ressum,rat
      Parameter(pi=3.14159265359d0)
      External hg2F1,gammln,Icos_km,ressum
      kappa0=prmt(1)
      nu_in=prmt(2)
      nu_out=prmt(3)
      rb=prmt(4)
      ellip=prmt(5)
      phi0=prmt(6)
      x=prmt(7)
      y=prmt(8)
      eps=prmt(9)
      mu=(nu_out-3.d0)/2.d0
      muinv=1.d0/mu
      e=ABS(ellip*(2.d0-ellip)/(1.d0+(1.d0-ellip)**2))
      if (abs(x) .lt. 1.d-10 .and. abs(y) .lt. 1.d-10) then
         r=0.d0
      else
         r=sqrt(x**2+y**2)
      end if
      phi=datan2(y,x)
*
* Lensing by a circularly symmetrical deflector
*----------------------------------------------
      IF (e .lt. 1.d-6) THEN
         t=(r/rb)**2
         Int0=(4.d0*(nu_out-nu_in)/
     a        ((nu_out-3.d0)*(3.d0-nu_in)*(nu_out-1.d0)))
     a      *EXP(gammln(nu_out/2.d0+0.5d0)+gammln(2.5d0-nu_in/2.d0)
     a           -gammln((nu_out-nu_in)/2.d0+1.d0))
     a     -(SQRT(pi)*nu_out/((nu_out-3.d0)*(nu_out-1.d0)))
     a      *EXP(gammln(nu_out/2.d0+0.5d0)-gammln(nu_out/2.d0+1.d0))
     a      *(1.d0+t)**(-(nu_out-3.d0)/2.d0)
     a      *hg2F1(nu_in/2.d0,(nu_out-3.d0)/2.d0,nu_out/2.d0,
     a             1.d0/(1.d0+t),eps)
         Int0=2.d0*kappa0*(rb**2/r)*Int0
         Int0p=2.d0*kappa0*r*SQRT(pi)*(nu_out/(nu_out-1.d0))
     a      *EXP(gammln(nu_out/2.d0+0.5d0)-gammln(nu_out/2.d0+1.d0))
     a      *(1.d0+t)**(-(nu_out-1.d0)/2.d0)
     a      *hg2F1(nu_in/2.d0,(nu_out-1.d0)/2.d0,nu_out/2.d0,
     a             1.d0/(1.d0+t),eps)
         cphi=cos(phi)
         sphi=sin(phi)
         ALr=Int0
         ALx=ALr*cphi
         ALy=ALr*sphi
         ALrr=(-Int0+Int0p)/r
         PSI11=cphi**2*ALrr+sphi**2*ALr/r
         PSI22=sphi**2*ALrr+cphi**2*ALr/r
         PSI12=sphi*cphi*ALrr-sphi*cphi*ALr/r
         PSI21=PSI12
         go to 100
      END IF
*
* Hereafter:lensing by an elliptical mass distribution
*-----------------------------------------------------    
      P=1.d0/rb**2
      Q=P*e
      f1=1.d0+P*r**2
      f2=sqrt(f1**2-(Q*r**2)**2)      
      h=r/f2
      e1=abs((f1-f2)/(f1+f2))
      e1sqrt=sqrt(e1)
      e2=abs(.5d0*(f1/f2**2+1.d0/f2))
      e2mu=e2**mu
*
*---------------------------------------------------------*
* Evaluation of F(m-mu,-mu;m+1;e1) & F(m-1-mu,-mu;m+1;e1) *
* for m = 0 to mmax (initial estimate).                   *
*---------------------------------------------------------*
* Determine an initial estimate of mmax:
      C4=1.d0
      sum_1=1.d0
      t1=1.d0
      m=0
 3    sum0_1=sum_1
      m=m+1
      dpm=DBLE(m)
      C4=C4*(-mu+dpm-1.d0)/dpm
      t1=t1*e1sqrt
      sum_1=sum_1+C4*t1
      if (ABS((sum_1-sum0_1)/sum_1) .gt. eps) go to 3
      mmax=m+INT(0.3*m)
*
      dpm=DBLE(mmax)
      hgt1=hg2F1(dpm+1.d0-mu,-mu,dpm+2.d0,e1,eps)
      hgt2=hg2F1(dpm-mu,-mu,dpm+2.d0,e1,eps)
      Fval_muk0(mmax+1)=hgt1
      Fval_muk0a(mmax+1)=hgt2
      a=dpm+1.d0-mu
      b=-mu
      c=dpm+2.d0
      do m=mmax,0,-1
      hgt3=hgt1*(a-1.d0)/(c-1.d0)+hgt2*(c-a)/(c-1.d0)
      hgt2=hgt2*e1*(c-b-1.d0)/(c-1.d0)+(1.d0-e1)*hgt3
      hgt1=hgt3
      a=a-1.d0
      c=c-1.d0
      Fval_muk0(m)=hgt3
      Fval_muk0a(m)=hgt2
      enddo
      mmax=mmax+1
*
*---------------------------------*
* Evaluation of Int0-prime & Int0 *
*---------------------------------*
      hgt1=-Fval_muk0a(0)*(1.d0+mu)*muinv/(1.d0-e1)
     a     +Fval_muk0(0)*(2.d0*mu+1.d0)*muinv/(1.d0-e1)
      hgt2=Fval_muk0(0)
      a=-mu+1.d0
      b=-mu
      c=1.d0
      CC0=(2.d0*sqrt(pi)/(nu_out-1.d0))
     a    *EXP(gammln((nu_out+1.d0)/2.d0)-gammln(nu_out/2.d0))
      Cval(0)=CC0
      CC=CC0
      C1=CC/mu
      sum_1=hgt2*C1
      sum_3=hgt2*CC
      t=1.d0
      k=0
 9    sum0_1=sum_1
      sum0_3=sum_3
      k=k+1
      dpk=DBLE(k)
      CC=CC*(nu_in/2.d0+dpk-1.d0)*(nu_out/2.d0+dpk-1.5d0)
     a     /((nu_out/2.d0+dpk-1.d0)*dpk)
      Cval(k)=CC
      cmax=k
      C1=C1+CC/(dpk+mu)
      hgt3=(a-1.d0)/(c-b)*(1.d0-e1)*hgt1
     a    +((b-1.d0)*(c-b)-(a-1.d0)*(c-a))/((b-a)*(c-b))*hgt2
      hgt2=(c+2.d0-2.d0*a-(b-a)*e1)/(c-a+1.d0)*hgt3
     a    -((a-1.d0)*(b-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)**2*hgt1
     a    +((a-1.d0)*(c-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)*hgt2
      hgt1=hgt3
      a=a-1.d0
      b=b-1.d0
      t=t*e2
      sum_1=sum_1+t*hgt2*C1
      sum_3=sum_3+t*hgt2*CC
      fracmax=MAX(ABS((sum_1-sum0_1)/sum_1),ABS((sum_3-sum0_3)/sum_3))
      if (fracmax .gt. eps) go to 9
      Int0=-h*e2mu*sum_1+((rb**2/r)/SQRT(1.d0-e**2))*8.d0
     a    *((nu_out-nu_in)/((nu_out-3.d0)*(nu_out-1.d0)*(3.d0-nu_in)))
     a    *EXP(gammln((nu_out+1.d0)/2.d0)+gammln((5.d0-nu_in)/2.d0)
     a        -gammln((nu_out-nu_in+2.d0)/2.d0))
      Int0p=2.d0*h*e2mu*sum_3
      Int1(0)=Int0
      Int2(0)=-Int0
      Int3(0)=Int0p/2.d0
      IF (r .gt. 1.3d0*rb/sqrt(1.d0-e)) THEN
*
*----------------------------------------------------------------*
* Evaluation of Int1, Int2 & Int3: keep Int1(1) through Int1(m1) *
*                                  keep Int2(1) through Int2(m1) *
*                                  keep Int3(1) through Int3(m1) *
*----------------------------------------------------------------*
      m=0
      C2=muinv
      C4=1.d0
      C6=muinv
      C8=1.d0
      sum_int1=ABS(Int1(0))
      sum_int2=0.d0
      sum_int3=ABS(Int3(0))
 20   sum0_int1=sum_int1
      sum0_int2=sum_int2
      sum0_int3=sum_int3
      m=m+1
      dpm=DBLE(m)
      C2=C2/((mu-dpm)*dpm)
      C4=C4*(dpm+mu)*(-dpm+mu+1.d0)
      C6=C6*(dpm-1.d0+mu)/dpm
      C8=C8*2.d0*(2.d0*dpm-1.d0)
      if (m .le. mmax) then
         hgt1=Fval_muk0(m)
         hgt2=Fval_muk0a(m)
      else
         hgt1=hg2F1(dpm-mu,-mu,dpm+1.d0,e1,eps)
         hgt2=hg2F1(dpm-1.d0-mu,-mu,dpm+1.d0,e1,eps)
         mmax=m
      end if
      hgt1p=1.d0
      hgt2p=1.d0+e1*dpm/(dpm+1.d0)
      a=dpm-mu
      b=-mu
      c=dpm+1.d0
      ap=0.d0
      bp=-dpm
      cp=dpm+1.d0
      CC=CC0
      C3=C2
      C5=C4
      C7=C6
      C9=C8
      t=1.d0
      D1=CC*C3
      D2=CC*C7
      C1=(dpm+mu)*C7
      sum_11=C5*hgt1*D1
      sum_12=C9*hgt1p
      sum_2=hgt1*D2
      sum_3=CC*C1*hgt1
      k=0
 21   sum0_11=sum_11
      sum0_2=sum_2
      sum0_3=sum_3
      k=k+1
      dpk=DBLE(k)
      hgt3=(a-1.d0)/(c-b)*(1.d0-e1)*hgt1
     a    +((b-1.d0)*(c-b)-(a-1.d0)*(c-a))/((b-a)*(c-b))*hgt2
      hgt2=(c+2.d0-2.d0*a-(b-a)*e1)/(c-a+1.d0)*hgt3
     a    -((a-1.d0)*(b-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)**2*hgt1
     a    +((a-1.d0)*(c-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)*hgt2
      hgt1=hgt3
      a=a-1.d0
      b=b-1.d0
      t=t*e2
      if (k .le. cmax) then
        CC=Cval(k)
      else
        CC=Cval(cmax)*(nu_in/2.d0+dpk-1.d0)*(nu_out/2.d0+dpk-1.5d0)
     a     /((nu_out/2.d0+dpk-1.d0)*dpk)
        cmax=cmax+1
        Cval(cmax)=CC
      endif
      C3=C3*(1.d0-(dpm+1.d0)/(dpk+mu))
      D1=D1+CC*C3
      C5=C5*(dpm+dpk+mu)/(-dpm+dpk+mu)
      C7=C7*(1.d0+(dpm-1.d0)/(dpk+mu))
      D2=D2+CC*C7
      C1=(dpm+dpk+mu)*C7
      sum_11=sum_11+C5*t*hgt3*D1
      sum_2=sum_2+t*hgt3*D2
      sum_3=sum_3+CC*C1*t*hgt3
      fracmax=MAX(ABS((sum_2-sum0_2)/sum_2)
     a           ,ABS((sum_11-sum0_11)/sum_11))
      fracmax=MAX(fracmax,ABS((sum_3-sum0_3)/sum_3))
      if (fracmax .gt. eps) go to 21
      t1=(-e1sqrt)**m*h*e2mu
      Int2(m)=t1*sum_2
      Int3(m)=t1*sum_3
      if (ABS(e2**(dpm-mu)*dpm*C8/C4) .gt. eps) then
      k=0
      t=1.d0
 22   sum0_12=sum_12
      k=k+1
      dpk=DBLE(k)
      hgt3p=(ap-1.d0)/(cp-bp)*(1.d0-e1)*hgt1p
     a +((bp-1.d0)*(cp-bp)-(ap-1.d0)*(cp-ap))/((bp-ap)*(cp-bp))*hgt2p
      hgt2p=(cp+2.d0-2.d0*ap-(bp-ap)*e1)/(cp-ap+1.d0)*hgt3p
     a  -((ap-1.d0)*(bp-ap))/((cp-ap+1.d0)*(cp-bp))*(1.d0-e1)**2*hgt1p
     a  +((ap-1.d0)*(cp-ap))/((cp-ap+1.d0)*(cp-bp))*(1.d0-e1)*hgt2p
      hgt1p=hgt3p
      t=t*e2
      C9=C9*((dpk+2.d0*dpm)/dpk)
      ap=ap-1.d0
      bp=bp-1.d0
      sum_12=sum_12+C9*t*hgt3p
      if (abs((sum_12-sum0_12)/sum_12) .gt. eps) go to 22
      end if
      sum_12=sum_12*pi**1.5d0*((nu_out-nu_in)/2.d0)
     a      /SIN(pi*(nu_out-3.d0-2.d0*dpm)/2.d0)
     a      *EXP(gammln((3.d0+2.d0*dpm-nu_in)/2.d0)
     a          -gammln((5.d0+2.d0*dpm-nu_out)/2.d0)
     a          -gammln((nu_out-nu_in+2.d0)/2.d0)
     a          -gammln(1.5d0+dpm))
      Int1(m)=-t1*sum_11+t1*e2**(dpm-mu)*sum_12
      sum_int1=sum_int1+ABS(dpm*Int1(m))
      sum_int2=sum_int2+ABS(dpm*Int2(m))
      sum_int3=sum_int3+ABS(dpm*Int3(m))
      fracmax=MAX(ABS((sum_int2-sum0_int2)/sum_int2)
     a           ,ABS((sum_int1-sum0_int1)/sum_int1))
      fracmax=MAX(fracmax,ABS((sum_int3-sum0_int3)/sum_int3))
      if (fracmax .gt. eps) go to 20
      m1=m
      m2=m
      m3=m
*
      ELSE
      eps=eps*1.d-1
*----------------------------------------------------------*
* Evaluation of Int2 & Int3: keep Int2(1) through Int2(m2) *
*                            keep Int3(1) through Int3(m3) *
*----------------------------------------------------------*
      m=0
      C6=muinv
      sum_int2=0.d0
      sum_int3=0.d0
 30   sum0_int2=sum_int2
      sum0_int3=sum_int3
      m=m+1
      dpm=DBLE(m)
      klim=2+INT(dpm*e2/(1.d0-e2))
     a     +2*INT(LOG(eps)/LOG(0.5d0+0.5d0*e2))
      C6=C6*(dpm-1.d0+mu)/dpm
      if (m .le. mmax) then
         hgt1=Fval_muk0(m)
         hgt2=Fval_muk0a(m)
      else
         hgt1=hg2F1(dpm-mu,-mu,dpm+1.d0,e1,eps)
         hgt2=hg2F1(dpm-1.d0-mu,-mu,dpm+1.d0,e1,eps)
         mmax=m
         Fval_muk0(m)=hgt1
         Fval_muk0a(m)=hgt2
      end if
      a=dpm-mu
      b=-mu
      c=dpm+1.d0
      CC=CC0
      C7=C6
      t=1.d0
      D2=CC*C7
      C1=(dpm+mu)*C7
      sum_2=hgt1*D2
      sum_3=CC*C1*hgt1
      k=0
 31   sum0_2=sum_2
      sum0_3=sum_3
      k=k+1
      dpk=DBLE(k)
      hgt3=(a-1.d0)/(c-b)*(1.d0-e1)*hgt1
     a    +((b-1.d0)*(c-b)-(a-1.d0)*(c-a))/((b-a)*(c-b))*hgt2
      hgt2=(c+2.d0-2.d0*a-(b-a)*e1)/(c-a+1.d0)*hgt3
     a    -((a-1.d0)*(b-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)**2*hgt1
     a    +((a-1.d0)*(c-a))/((c-a+1.d0)*(c-b))*(1.d0-e1)*hgt2
      hgt1=hgt3
      a=a-1.d0
      b=b-1.d0
      t=t*e2
      if (k .le. cmax) then
        CC=Cval(k)
      else
        CC=Cval(cmax)*(nu_in/2.d0+dpk-1.d0)*(nu_out/2.d0+dpk-1.5d0)
     a     /((nu_out/2.d0+dpk-1.d0)*dpk)
        cmax=cmax+1
        Cval(cmax)=CC
      endif
      C7=C7*(1.d0+(dpm-1.d0)/(dpk+mu))
      D2=D2+CC*C7
      C1=(dpm+dpk+mu)*C7
      sum_2=sum_2+t*hgt3*D2
      sum_3=sum_3+CC*C1*t*hgt3
      fracmax=MAX(ABS((sum_2-sum0_2)/sum_2)
     a           ,ABS((sum_3-sum0_3)/sum_3))
      if (k .lt. klim) then
      if (fracmax .gt. eps) go to 31
      endif
      t1=(-e1sqrt)**m*h*e2mu
      Int2(m)=t1*sum_2
      Int3(m)=t1*sum_3
      sum_int2=sum_int2+ABS(dpm*Int2(m))
      sum_int3=sum_int3+ABS(dpm*Int3(m))
      fracmax=MAX(ABS((sum_int2-sum0_int2)/sum_int2)
     a           ,ABS((sum_int3-sum0_int3)/sum_int3))
      if (fracmax .gt. eps) go to 30
      m2=m
      m3=m
*
*---------------------------------------------------*
* Evaluation of Int1: keep Int1(1) through Int1(m1) *
*---------------------------------------------------*
      t=(r/rb)**2/(1.d0+(r/rb)**2)
      et=e*t
      etsq=et**2
      t1=1.d0-t
      klim=2+ABS(INT(LOG(eps)/LOG(0.5d0+0.5d0*et)))
      m=0
      C1=1.d0
      sum_int1=ABS(Int0)
 35   m=m+1
      sum0_int1=sum_int1
      dpm=DBLE(m)
      C1=C1*(1.d0+mu/dpm)
      sum_1=0.d0
      k=-2
 36   k=k+2
      sum0_1=sum_1
      dpk=DBLE(k)
      if (k .eq. 0) then
        C2=C1
        t3=1.d0
      else
        C2=C2*(1.d0+mu/(dpk+dpm))*(1.d0+mu/(dpk-1.d0+dpm))
        t3=t3*etsq
      endif
      sum=0.d0
      l=-1
 37   l=l+1
      dpl=DBLE(l)
      sum0=sum
      if (l .le. cmax) then
        CC=Cval(l)
      else
        CC=Cval(cmax)*(nu_in/2.d0+dpl-1.d0)*(nu_out/2.d0+dpl-1.5d0)
     a     /((nu_out/2.d0+dpl-1.d0)*dpl)
        cmax=cmax+1
        Cval(cmax)=CC
      endif
      if (l .eq. 0) then
        C3=C2
        a=dpk+dpm+mu+1.d0
        b=1.d0
        c=dpk+2.d0*dpm+2.d0
        hgt1=hg2F1(a-1.d0,b,c,t,eps)
        hgt2=hg2F1(a,b,c,t,eps)
        hgt1p=hgt1/t1
        hgt2p=hgt2
      else
        C3=C3*(1.d0+(dpk+dpm)/(dpl+mu))
        hgt3p=((c-a)*t1*hgt1p+(2.d0*a-c-a*t+b*t)*hgt2p)/a
        hgt1p=hgt2p
        hgt2p=hgt3p
        a=a+1.d0
      endif
      term(l)=CC*C3*hgt2p
      sum=sum+term(l)
      if (l .le. 20) go to 37
      if (l .le. 800) then
        if (ABS((sum-sum0)/sum) .lt. eps*1.d-1) then
          go to 40
        else  
          go to 37
        endif
      endif
      t2=(dpl-1.d0)*(term(l-1)/term(l)-1.d0)
      sum=sum+term(l)*ressum(l,t2)
 40   sum_1=sum_1+(2.d0/(dpk+2.d0*dpm+1.d0))*t3*Icos_km(k+m,m)*sum
      if (k .le. klim) then
      if (ABS((sum_1-sum0_1)/sum_1) .gt. eps) go to 36
      endif
      Int1(m)=(r/pi)*(-et)**m*t1**(mu+1.d0)*sum_1
      if (m .eq. 1) then 
        rat=0.9d0
      else
        rat=ABS(Int1(m)/Int1(m-1))
      endif
      if (rat .gt. 1.d0) Int1(m)=0.d0
      sum_int1=sum_int1+ABS(dpm*Int1(m))
      fracmax=ABS((sum_int1-sum0_int1)/sum_int1)
      if (fracmax .gt. eps) go to 35
      m1=m
      END IF
*-----------------------------------------------*
*Calculation of the Deflection Angle Components *
*-----------------------------------------------*
      if (m1 .gt. m2) then
        do m=m2+1,m1
        Int2(m)=0.d0
        enddo
      else if (m2 .gt. m1) then
        do m=m1+1,m2
        Int1(m)=0.d0
        enddo
        m1=m2
      endif
      phi1=2.d0*(phi-phi0)
      cphi=cos(phi)
      sphi=sin(phi)
      cphi1(1)=cos(phi1)
      sphi1(1)=sin(phi1)
      if (m1 .gt. 1) then
      do m=2,m1
      cphi1(m)=cphi1(1)*cphi1(m-1)-sphi1(1)*sphi1(m-1)
      sphi1(m)=sphi1(1)*cphi1(m-1)+cphi1(1)*sphi1(m-1)
      enddo
      end if
*:Alpha(r) and Alpha(phi)
      ALr=Int0
      ALphi=0.d0
      do m=1,m1
      ALr=ALr+cphi1(m)*(Int1(m)-Int2(m))
      ALphi=Alphi+sphi1(m)*(Int1(m)+Int2(m))
      enddo
      ALr=kappa0*ALr
      ALphi=kappa0*ALphi
*:transformation to Alpha(x) and Alpha(y)
      ALx=ALr*cphi-ALphi*sphi
      ALy=ALr*sphi+ALphi*cphi
*----------------------------------------*
*Calculation of the magnification matrix *
*----------------------------------------*
      ALrr=-Int0+Int0p
      ALrp=0.d0
      ALpr=0.d0
      ALpp=0.d0
      do m=1,m1
      ALrr=ALrr-cphi1(m)*((2*m+1)*Int1(m)+(2*m-1)*Int2(m))
      ALrp=ALrp-2.d0*sphi1(m)*m*(Int1(m)-Int2(m))
      ALpr=ALpr-sphi1(m)*((2*m+1)*Int1(m)-(2*m-1)*Int2(m))
      ALpp=ALpp+2.d0*cphi1(m)*m*(Int1(m)+Int2(m))
      enddo
      do m=1,m3
      ALrr=ALrr+4.d0*cphi1(m)*Int3(m)
      enddo
      ALrr=ALrr*(kappa0/r)
      ALrp=ALrp*(kappa0/r)
      ALpr=ALpr*(kappa0/r)
      ALpp=ALpp*(kappa0/r)
      PSI11=cphi**2*ALrr-cphi*sphi*ALpr+sphi**2*ALr/r
     a     -sphi*cphi*ALrp+sphi*cphi*ALphi/r+sphi**2*ALpp
      PSI22=sphi**2*ALrr+sphi*cphi*ALpr+cphi**2*ALr/r
     a     +sphi*cphi*ALrp-sphi*cphi*ALphi/r+cphi**2*ALpp
      PSI12=sphi*cphi*ALrr-sphi**2*ALpr-sphi*cphi*ALr/r
     a     +cphi**2*ALrp-cphi**2*ALphi/r-sphi*cphi*ALpp
      PSI21=PSI12
 100  dfl(1)=ALx
      dfl(2)=ALy
      mgnf(1)=1.d0-PSI11
      mgnf(2)=-PSI12
      mgnf(3)=-PSI21
      mgnf(4)=1.d0-PSI22
      zeta=MAX(abs(mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3)),1.d-6)
      zeta=DSIGN(zeta,mgnf(1)*mgnf(4)-mgnf(2)*mgnf(3))
      mgnf(5)=1.d0/zeta          
      return
*
      END  
******************************************************************************
      FUNCTION ressum(n0,a)
      Implicit none
      Integer n0,n
      Double precision ressum,sum,a,u,u1,dpn
      u=1.d0
      sum=0.d0
      do n=n0-1,1,-1
        dpn=DBLE(n)
        u=u*(a/dpn+1.d0)
        sum=sum+u
      enddo
      u1=u
      ressum=u1*a/(a-1.d0)-sum
      return
      END
